"""Main module for the game"""

import pygame
import os
import time
import logging

import serge.engine
import serge.world
import serge.actor
import serge.zone
import serge.render
import serge.sound 
import serge.events
import serge.blocks.achievements
try:
    import serge.builder.builder
except ImportError:
    pass

import serge.blocks.utils

import common
import startscreen
import creditsscreen

from theme import G, theme

def registerSounds():
    """Register the sounds to use"""
    serge.sound.Sounds.setPath('sound')
    r = serge.sound.Sounds.registerItem
    r('click', '102130__noirpantalon__hard-subby-kick.wav')
    r('placed', '30341__junggle__waterdrop24.wav')
    r('damage', '67258__aent__match.wav')
    
def registerMusic():
    """Register the music to use"""
    serge.sound.Music.setPath('music')
    r = serge.sound.Music.registerItem
    r('intro', 'Dissolved_-_12_-_The_Spilling_Reef-a.ogg')
    r('track-1', 'Mobster_-_04_-_Earthbreaker.mp3')
    r('track-2', 'Mobster_-_09_-_End_of_machine_age.mp3')
    r('track-3', 'Mobster_-_12_-_Hive.mp3')
    
def registerGraphics():
    """Register the graphics to use"""
    serge.visual.Sprites.setPath('graphics')
    r = serge.visual.Sprites.registerItem
    rf = serge.visual.Sprites.registerFromFiles
    rm = serge.visual.Sprites.registerMultipleItems
    rp = serge.visual.Sprites.registerItemsFromPattern
    r('icon', 'icon.png')
    r('logo', 'logo.png')
    r('lock', 'lock.png')
    rf('mute-button', 'music-%d.png', 2, zoom=0.5)
    r('check-mark', 'check-mark.png')
    rm(['road', 'wall', 'surface', 'start', 'near-start', 'goal', 'x1', 'x2', 'ground', 'goal-1', 'goal-2', 'goal-3'], 
        'tileset-2.png', 3, 4)
    rp('\w+-bullet.png')
    rf('turret-1', 'turret-1-%d.png', 4, running=False)
    rf('turret-2', 'turret-2-%d.png', 4, running=False)
    rf('turret-3', 'turret-3-%d.png', 4, running=False)
    rf('turret-4', 'turret-4-%d.png', 4, running=False)
    rf('turret-5', 'turret-5-%d.png', 4, running=False)
    r('turret-bad', 'turret-bad.png')
    rp('enemy-\d.png')
    rp('enemy-\d-pain.png')
    r('fire', 'fire.png', 9, 1, 30, running=True, loop=False)
    r('ice', 'ice-tileset.png', 4, running=False)
    r('turret-6', 'ice.png')
    r('ice-bad', 'ice-bad.png')
    r('bg-1', 'bg-1.png')
    r('start-bg', 'start-bg.png')
    r('blood', 'blood-1.png')
    r('default-bg', 'default-bg.png')
    rp('level-\d*-bg.png')
    rp('level-\d*-fg.png')
    rp('hint\d.png')
    r('credits', 'credits.png')
    rf('achievement', 'achievement-%d.png', 2)
    r('pain', 'pain.png')
    #
    serge.visual.Fonts.setPath('fonts')
    serge.visual.Fonts.registerItem('DEFAULT', 'AveriaSerif-Bold.ttf')
    
def registerEvents():
    """Register all the events"""
    broadcaster = serge.events.getEventBroadcaster()
    broadcaster.registerEventsFromModule(common)

def registerAchievements(options):
    """Register the achievements for the game"""
    r = serge.blocks.achievements.initManager('exogene').safeRegisterAchievement
    a = serge.blocks.achievements.Achievement
    r(a('Training Wheels', 'Completed the training level', 
        'badge', False, 'level-complete', condition_string='level, health, time, cash: level == 1'))
    r(a('Perfect finish', 'Complete a level without losing any health',
        'badge', False, 'level-complete', condition_string='level, health, time, cash: level != 1 and health == 100'))
    r(a('Living Dangerously', 'Complete a level with almost no health left',
        'badge', False, 'level-complete', condition_string='level, health, time, cash: level != 1 and health <= 20'))
    r(a('Quick Draw', 'Finish a level in less than 80 seconds',
        'badge', False, 'level-complete', condition_string='level, health, time, cash: level != 1 and time <= 80'))
    r(a('Slow Poke', 'Take 4 minutes to complete a level',
        'badge', False, 'level-complete', condition_string='level, health, time, cash: level != 1 and time >= 240'))
    r(a('Bank That', 'Finish a level with more than $1,000 left',
        'badge', False, 'level-complete', condition_string='level, health, time, cash: level != 1 and cash >= 1000'))
    r(a('Super Saver', 'Finish a level with more than $3,000 left',
        'badge', False, 'level-complete', condition_string='level, health, time, cash: level != 1 and cash >= 3000'))
    r(a('High Earner', 'Have more than $1,500 cash during a level',
        'badge', False, 'playing', condition_string='level, time, cash: level != 12 and cash >= 1500'))
    r(a('Equal Opportunist', 'Use all the different turret types in a level',
        'badge', False, 'placed', condition_string='placed_types, number: len(placed_types) == 4'))
    r(a('Scatter Shot', 'Use more than 20 turrets in a level',
        'badge', False, 'placed', condition_string='placed_types, number: number >= 20'))

    serge.blocks.achievements.addAchievementsWorld(options, theme)
    
def startEngine(options):
    """Start the main engine"""
    engine = serge.engine.Engine(width=G('screen-width'), height=G('screen-height'), title=G('screen-title'), icon=G('screen-icon'))
    serge.blocks.utils.createVirtualLayersForEngine(engine, ['board', 'background', 'hints', 'enemies', 'turrets', 'ice', 
        'foreground', 'ui-background', 'ui-middleground', 'ui', 'ui-buttons', 'front-ui'])
    serge.blocks.utils.createWorldsForEngine(engine, ['start-screen', 'main-screen', 'credit-screen'])
    #
    engine.setCurrentWorldByName('start-screen')
    return engine

        
def stoppingNow(obj, arg):
    """We are about to stop"""
    #
    # Fade out music and wait for a bit before going away
    serge.sound.Music.fadeout(G('pre-stop-pause')*1000)
    time.sleep(G('pre-stop-pause'))


def main(options, args):
    """Start the engine and the game"""
    registerSounds()
    registerMusic()
    registerGraphics()
    registerEvents()
    #
    # Change theme settings
    if options.theme:
        theme.updateFromString(options.theme)
    #
    if not options.log:
        serge.common.logger.setLevel(logging.ERROR)
    #
    # Create the engine
    engine = startEngine(options)
    engine.linkEvent(serge.events.E_BEFORE_STOP, stoppingNow)
    #
    # Muting
    mute = serge.blocks.actors.MuteButton('mute-button', 'ui', alpha=G('mute-button-alpha'))
    serge.blocks.utils.addMuteButtonToWorlds(mute, center_position=G('mute-button-position'))
    #
    if options.musicoff:
        mute.toggleSound()
    #
    # Initialise the main logic
    s = startscreen.main(options)
    creditsscreen.main(options)
    registerAchievements(options)
    #
    if options.movie:
        serge.blocks.utils.RecordDesktop(options.movie)
    #   
    if options.debug:
        serge.builder.builder.main(engine, options.framerate)
    else:
        engine.run(options.framerate)
